<?php
//=======================================================================
// File:        class.File.php
// Description: Represents a file on disk.
// Dependencies: DataException class
// Global references: $_FILES (static method UploadFile)
//=======================================================================

class File {
	private $fileName;

	/*-----------------------------------------------------------------------------
	 * $_sFile (string) path to the file
	 _____________________________________________________________________________*/
	function __construct($_sFile){
		if (!file_exists($_sFile))
			throw new InvalidArgumentException(__METHOD__ .' File ('. $_sFile .') does not exist.');

		if (is_dir($_sFile))
			throw new InvalidArgumentException(__METHOD__ .' Passed argument ('. $_sFile .') is not a file.');

		$this->fileName = $_sFile;
	}


	/*-----------------------------------------------------------------------------
	 * getSize()
	 * returns file size in bytes (int)
	 _____________________________________________________________________________*/
	public function getSize(){
		return filesize($this->fileName);
	}


	/*-----------------------------------------------------------------------------
	 * getModifiedTime()
	 * returns file modification time (Unix timestamp)
	 _____________________________________________________________________________*/
	public function getModifiedTime(){
		return filemtime($this->fileName);
	}


	/*-----------------------------------------------------------------------------
	 * getName()
	 * Returns filename component of path
	 _____________________________________________________________________________*/
	public function getName(){
		return basename($this->fileName);
	}


	/*-----------------------------------------------------------------------------
	 * getFullName
	 * Returns entire path to file and the file name Example: /usr/www/html/test/a.txt
	 _____________________________________________________________________________*/
	public function getFullName(){
		return $this->fileName;
	}
	
	/*------------------------------------------------------------------------------
	 * Copy file from one destination to another
	 * Returns success or failure
	_______________________________________________________________________________*/
	public function copyFile($source,$destination)
	$sourcefile = $source;
	$destfile = $destination;

	if (!copy($sourcefile, $destfile)) {
		//echo "failed to copy $file...\n";
		return false;
	} 
	else
	{
		return true;
	}
	
	/*------------------------------------------------------------------------------
	 * Copy file from one destination to another
	 * Returns success or failure
	_______________________________________________________________________________*/
	public function deleteFile($fileNameToDelete)
	
	if (!unlink($fileNameToDelete)) {
		return false;
	} 
	else
	{
		return true;
	}
	
	
	/*-----------------------------------------------------------------------------
	 * FormatFileSize()
	 * Formats file size (number) in Bytes into a human readable format.
	 * $iSize - (int) file size in Bytes
 	 * Returns string
	 _____________________________________________________________________________*/
	static function FormatFileSize($iSize){
		$iSize = intval($iSize);
		if ($iSize < 1){
            return;
        }

		if ($iSize < 1024){
		 	 return (string)$iSize  .' Bytes';
		}elseif($iSize < 1024*1024){
    		return round($iSize/1024,2) .' KB';
		}else{
		 	 $iSize /= 1048576;
        }

        return  round($iSize,2)  .' MB';
	}


	/*-----------------------------------------------------------------------------
	 * CleanFileName
	 * Returns a string where any character not matching [a-zA-Z0-9_-] is substituted by '_'
	 _____________________________________________________________________________*/
	static function CleanFileName($fileName){
		$theNewName = ereg_replace("[^\.[:alnum:]_-]",'_',trim($fileName));
		return $theNewName;
	}


	/*-----------------------------------------------------------------------------
	 * UploadFile()
	 * handles file upload
	 * 
	 * $uploadName - (string) name of the html file input element <input type="file" name='{$uploadName}'
	 * $sDestDir - (string) destination directory on disk (where to upload); If it is an empty string   
	 *                      the file will be leaved (might be useful when we just want to validate file 
	 *                      and leave it in the temp upload folder)
	 * $maxAllowedSize - (int) optional - it can validate the size of the uploaded file and disallow to upload 
	 *                         files bigger that $maxAllowedSize Bytes
	 * $aAllowedType - (array) optional - array of mime types - it can validate the file type and disallow to 
	 *                         upload files that have disallowed mime type
	 *
	 * returns array of File objects if succeded (or null if $sDestDir vas empty)
	 * Global reference: $_FILES - HTTP File upload variables
	 _____________________________________________________________________________*/
	 static function UploadFile($uploadName, $sDestDir, $maxAllowedSize = null, $aAllowedType = null){
        $aRetFilesCol = array();
        $uploadedFiles = array();
        $nbOfUploadedFiles = 1;

        //__/ VALIDATE UPLOADED FILE
        if (!is_array($_FILES) || count($_FILES) == 0){
            throw new Exception('No file was uploaded for an unknown reason. Did you want to upload a very large file? Try to upload a smaller file or compress the file to reduce its size.');
        }

        if (trim($sDestDir)!=''){
            //if (!is_dir($sDestDir))
                //throw new CException('File upload error: destination directory does not exist. '. $sDestDir, NULL, 'File::__UploadFile() $sDestDir='. $sDestDir);
        }


        // --
        if (gettype(($_FILES[$uploadName]['error']) == 'array')){
            $nbOfUploadedFiles = count($_FILES[$uploadName]['error']);
        }

        if ($nbOfUploadedFiles === 1 && 
            gettype($_FILES[$uploadName]['error']) !== 'array'){

            $uploadedFiles[] = array('error'    => $_FILES[$uploadName]['error'],
                                     'type'     => $_FILES[$uploadName]['type'],
                                     'size'     => $_FILES[$uploadName]['size'],
                                     'name'     => $_FILES[$uploadName]['name'],
                                     'tmp_name' => $_FILES[$uploadName]['tmp_name']);

        }else{

            for($i=0; $i < $nbOfUploadedFiles; $i++){
                $uploadedFiles[] = array('error'    => $_FILES[$uploadName]['error'][$i],
                                         'type'     => $_FILES[$uploadName]['type'][$i],
                                         'size'     => $_FILES[$uploadName]['size'][$i],
                                         'name'     => $_FILES[$uploadName]['name'][$i],
                                         'tmp_name' => $_FILES[$uploadName]['tmp_name'][$i]);
            }
        }


        for($i=0; $i < count($uploadedFiles); $i++){
            $uploadedFile = $uploadedFiles[$i];


    		if (gettype($uploadedFile['error']) == 'integer'){
    			switch(intval($uploadedFile['error'])){
    				case 1: throw new Exception('The uploaded file exceeds the upload_max_filesize directive in php.ini. '. gettype($uploadedFile['error']));
    					break;
    				case 3: throw new Exception('The uploaded file was only partially uploaded.');
    					break;
    				case 4: throw new Exception('No file was uploaded.');
    					break;
    			}
    		}else{
                throw new Exception('error: '. gettype($_FILES[$uploadName]['error']) .' -- '. count($_FILES[$uploadName]['error']) );
            }

    		// validate MIME type
    		if (is_array($aAllowedType)){
    			if (!in_array($uploadedFile['type'], $aAllowedType))
    				throw new DataException('Could not upload file. Invalid file content type. (Allowed file types are: '. implode(',',$aAllowedType) .')');
    		}

    		// validate size
    		if(!is_null($maxAllowedSize)){
    			if ($uploadedFile['size'] > $maxAllowedSize)
    				throw new DataException('Could not upload file. Size of the uploaded file exceeds the maximum allowed size ('. File::FormatFileSize($maxAllowedSize) .').');
    		}


    		//__/ UPLOAD FILE (if required)
    		if (trim($sDestDir)!=''){
    			$sUploadFile = $sDestDir . File::CleanFileName($uploadedFile['name']);

    			try{
    				if (move_uploaded_file($uploadedFile['tmp_name'], $sUploadFile))
    					$aRetFilesCol[] = new File($sUploadFile);
//				else
//					throw new AccessException('Possible file upload attack!', 0);
    			} catch(Exception $e){
    				throw new RuntimeException($e->getMessage());
    			}
    		}

        }

		return $aRetFilesCol;
	}

}
